/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.connect.data;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.Temporal;
import java.util.Base64;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;
import org.apache.iceberg.FileFormat;
import org.apache.iceberg.Schema;
import org.apache.iceberg.Table;
import org.apache.iceberg.connect.IcebergSinkConfig;
import org.apache.iceberg.connect.data.SchemaUpdate;
import org.apache.iceberg.connect.data.SchemaUtils;
import org.apache.iceberg.data.GenericRecord;
import org.apache.iceberg.data.Record;
import org.apache.iceberg.mapping.MappedField;
import org.apache.iceberg.mapping.NameMapping;
import org.apache.iceberg.mapping.NameMappingParser;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.ByteBuffers;
import org.apache.iceberg.util.DateTimeUtil;
import org.apache.iceberg.util.UUIDUtil;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.ConnectException;

class RecordConverter {
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private static final DateTimeFormatter OFFSET_TIMESTAMP_FORMAT = new DateTimeFormatterBuilder().append(DateTimeFormatter.ISO_LOCAL_DATE_TIME).appendOffset("+HHmm", "Z").toFormatter(Locale.ROOT);
    private final Schema tableSchema;
    private final NameMapping nameMapping;
    private final IcebergSinkConfig config;
    private final Map<Integer, Map<String, Types.NestedField>> structNameMap = Maps.newHashMap();

    RecordConverter(Table table, IcebergSinkConfig config) {
        this.tableSchema = table.schema();
        this.nameMapping = this.createNameMapping(table);
        this.config = config;
    }

    Record convert(Object data) {
        return this.convert(data, null);
    }

    Record convert(Object data, SchemaUpdate.Consumer schemaUpdateConsumer) {
        if (data instanceof Struct || data instanceof Map) {
            return this.convertStructValue(data, this.tableSchema.asStruct(), -1, schemaUpdateConsumer);
        }
        throw new UnsupportedOperationException("Cannot convert type: " + data.getClass().getName());
    }

    private NameMapping createNameMapping(Table table) {
        String nameMappingString = (String)table.properties().get("schema.name-mapping.default");
        return nameMappingString != null ? NameMappingParser.fromJson((String)nameMappingString) : null;
    }

    private Object convertValue(Object value, Type type, int fieldId, SchemaUpdate.Consumer schemaUpdateConsumer) {
        if (value == null) {
            return null;
        }
        switch (type.typeId()) {
            case STRUCT: {
                return this.convertStructValue(value, type.asStructType(), fieldId, schemaUpdateConsumer);
            }
            case LIST: {
                return this.convertListValue(value, type.asListType(), schemaUpdateConsumer);
            }
            case MAP: {
                return this.convertMapValue(value, type.asMapType(), schemaUpdateConsumer);
            }
            case INTEGER: {
                return this.convertInt(value);
            }
            case LONG: {
                return this.convertLong(value);
            }
            case FLOAT: {
                return Float.valueOf(this.convertFloat(value));
            }
            case DOUBLE: {
                return this.convertDouble(value);
            }
            case DECIMAL: {
                return this.convertDecimal(value, (Types.DecimalType)type);
            }
            case BOOLEAN: {
                return this.convertBoolean(value);
            }
            case STRING: {
                return this.convertString(value);
            }
            case UUID: {
                return this.convertUUID(value);
            }
            case BINARY: {
                return this.convertBase64Binary(value);
            }
            case FIXED: {
                return ByteBuffers.toByteArray((ByteBuffer)this.convertBase64Binary(value));
            }
            case DATE: {
                return this.convertDateValue(value);
            }
            case TIME: {
                return this.convertTimeValue(value);
            }
            case TIMESTAMP: {
                return this.convertTimestampValue(value, (Types.TimestampType)type);
            }
        }
        throw new UnsupportedOperationException("Unsupported type: " + String.valueOf(type.typeId()));
    }

    protected GenericRecord convertStructValue(Object value, Types.StructType schema, int parentFieldId, SchemaUpdate.Consumer schemaUpdateConsumer) {
        if (value instanceof Map) {
            return this.convertToStruct((Map)value, schema, parentFieldId, schemaUpdateConsumer);
        }
        if (value instanceof Struct) {
            return this.convertToStruct((Struct)value, schema, parentFieldId, schemaUpdateConsumer);
        }
        throw new IllegalArgumentException("Cannot convert to struct: " + value.getClass().getName());
    }

    private GenericRecord convertToStruct(Map<?, ?> map, Types.StructType schema, int structFieldId, SchemaUpdate.Consumer schemaUpdateConsumer) {
        GenericRecord result = GenericRecord.create((Types.StructType)schema);
        map.forEach((recordFieldNameObj, recordFieldValue) -> {
            String recordFieldName = recordFieldNameObj.toString();
            Types.NestedField tableField = this.lookupStructField(recordFieldName, schema, structFieldId);
            if (tableField == null) {
                Type type;
                if (schemaUpdateConsumer != null && (type = SchemaUtils.inferIcebergType(recordFieldValue, this.config)) != null) {
                    String parentFieldName = structFieldId < 0 ? null : this.tableSchema.findColumnName(structFieldId);
                    schemaUpdateConsumer.addColumn(parentFieldName, recordFieldName, type);
                }
            } else {
                result.setField(tableField.name(), this.convertValue(recordFieldValue, tableField.type(), tableField.fieldId(), schemaUpdateConsumer));
            }
        });
        return result;
    }

    private GenericRecord convertToStruct(Struct struct, Types.StructType schema, int structFieldId, SchemaUpdate.Consumer schemaUpdateConsumer) {
        GenericRecord result = GenericRecord.create((Types.StructType)schema);
        struct.schema().fields().forEach(recordField -> {
            Types.NestedField tableField = this.lookupStructField(recordField.name(), schema, structFieldId);
            if (tableField == null) {
                if (schemaUpdateConsumer != null) {
                    String parentFieldName = structFieldId < 0 ? null : this.tableSchema.findColumnName(structFieldId);
                    Type type = SchemaUtils.toIcebergType(recordField.schema(), this.config);
                    schemaUpdateConsumer.addColumn(parentFieldName, recordField.name(), type);
                }
            } else {
                boolean hasSchemaUpdates = false;
                if (schemaUpdateConsumer != null) {
                    String fieldName;
                    Type.PrimitiveType evolveDataType = SchemaUtils.needsDataTypeUpdate(tableField.type(), recordField.schema());
                    if (evolveDataType != null) {
                        fieldName = this.tableSchema.findColumnName(tableField.fieldId());
                        schemaUpdateConsumer.updateType(fieldName, evolveDataType);
                        hasSchemaUpdates = true;
                    }
                    if (tableField.isRequired() && recordField.schema().isOptional()) {
                        fieldName = this.tableSchema.findColumnName(tableField.fieldId());
                        schemaUpdateConsumer.makeOptional(fieldName);
                        hasSchemaUpdates = true;
                    }
                }
                if (!hasSchemaUpdates) {
                    result.setField(tableField.name(), this.convertValue(struct.get(recordField), tableField.type(), tableField.fieldId(), schemaUpdateConsumer));
                }
            }
        });
        return result;
    }

    private Types.NestedField lookupStructField(String fieldName, Types.StructType schema, int structFieldId) {
        if (this.nameMapping == null) {
            return this.config.schemaCaseInsensitive() ? schema.caseInsensitiveField(fieldName) : schema.field(fieldName);
        }
        return (Types.NestedField)this.structNameMap.computeIfAbsent(structFieldId, notUsed -> this.createStructNameMap(schema)).get(fieldName);
    }

    private Map<String, Types.NestedField> createStructNameMap(Types.StructType schema) {
        HashMap map = Maps.newHashMap();
        schema.fields().forEach(col -> {
            MappedField mappedField = this.nameMapping.find(col.fieldId());
            if (mappedField != null && !mappedField.names().isEmpty()) {
                mappedField.names().forEach(name -> map.put(name, col));
            } else {
                map.put(col.name(), col);
            }
        });
        return map;
    }

    protected List<Object> convertListValue(Object value, Types.ListType type, SchemaUpdate.Consumer schemaUpdateConsumer) {
        Preconditions.checkArgument((boolean)(value instanceof List));
        List list = (List)value;
        return list.stream().map(element -> {
            int fieldId = ((Types.NestedField)type.fields().get(0)).fieldId();
            return this.convertValue(element, type.elementType(), fieldId, schemaUpdateConsumer);
        }).collect(Collectors.toList());
    }

    protected Map<Object, Object> convertMapValue(Object value, Types.MapType type, SchemaUpdate.Consumer schemaUpdateConsumer) {
        Preconditions.checkArgument((boolean)(value instanceof Map));
        Map map = (Map)value;
        HashMap result = Maps.newHashMap();
        map.forEach((k, v) -> {
            int keyFieldId = ((Types.NestedField)type.fields().get(0)).fieldId();
            int valueFieldId = ((Types.NestedField)type.fields().get(1)).fieldId();
            result.put(this.convertValue(k, type.keyType(), keyFieldId, schemaUpdateConsumer), this.convertValue(v, type.valueType(), valueFieldId, schemaUpdateConsumer));
        });
        return result;
    }

    protected int convertInt(Object value) {
        if (value instanceof Number) {
            return ((Number)value).intValue();
        }
        if (value instanceof String) {
            return Integer.parseInt((String)value);
        }
        throw new IllegalArgumentException("Cannot convert to int: " + value.getClass().getName());
    }

    protected long convertLong(Object value) {
        if (value instanceof Number) {
            return ((Number)value).longValue();
        }
        if (value instanceof String) {
            return Long.parseLong((String)value);
        }
        throw new IllegalArgumentException("Cannot convert to long: " + value.getClass().getName());
    }

    protected float convertFloat(Object value) {
        if (value instanceof Number) {
            return ((Number)value).floatValue();
        }
        if (value instanceof String) {
            return Float.parseFloat((String)value);
        }
        throw new IllegalArgumentException("Cannot convert to float: " + value.getClass().getName());
    }

    protected double convertDouble(Object value) {
        if (value instanceof Number) {
            return ((Number)value).doubleValue();
        }
        if (value instanceof String) {
            return Double.parseDouble((String)value);
        }
        throw new IllegalArgumentException("Cannot convert to double: " + value.getClass().getName());
    }

    protected BigDecimal convertDecimal(Object value, Types.DecimalType type) {
        BigDecimal bigDecimal;
        if (value instanceof BigDecimal) {
            bigDecimal = (BigDecimal)value;
        } else if (value instanceof Number) {
            Number num = (Number)value;
            Double dbl = num.doubleValue();
            bigDecimal = dbl.equals(Math.floor(dbl)) ? BigDecimal.valueOf(num.longValue()) : BigDecimal.valueOf(dbl);
        } else if (value instanceof String) {
            bigDecimal = new BigDecimal((String)value);
        } else {
            throw new IllegalArgumentException("Cannot convert to BigDecimal: " + value.getClass().getName());
        }
        return bigDecimal.setScale(type.scale(), RoundingMode.HALF_UP);
    }

    protected boolean convertBoolean(Object value) {
        if (value instanceof Boolean) {
            return (Boolean)value;
        }
        if (value instanceof String) {
            return Boolean.parseBoolean((String)value);
        }
        throw new IllegalArgumentException("Cannot convert to boolean: " + value.getClass().getName());
    }

    protected String convertString(Object value) {
        try {
            if (value instanceof String) {
                return (String)value;
            }
            if (value instanceof Number || value instanceof Boolean) {
                return value.toString();
            }
            if (value instanceof Map || value instanceof List) {
                return MAPPER.writeValueAsString(value);
            }
            if (value instanceof Struct) {
                Struct struct = (Struct)value;
                byte[] data = this.config.jsonConverter().fromConnectData(null, struct.schema(), (Object)struct);
                return new String(data, StandardCharsets.UTF_8);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        throw new IllegalArgumentException("Cannot convert to string: " + value.getClass().getName());
    }

    protected Object convertUUID(Object value) {
        UUID uuid;
        if (value instanceof String) {
            uuid = UUID.fromString((String)value);
        } else if (value instanceof UUID) {
            uuid = (UUID)value;
        } else {
            throw new IllegalArgumentException("Cannot convert to UUID: " + value.getClass().getName());
        }
        if (FileFormat.PARQUET.name().toLowerCase(Locale.ROOT).equals(this.config.writeProps().get("write.format.default"))) {
            return UUIDUtil.convert((UUID)uuid);
        }
        return uuid;
    }

    protected ByteBuffer convertBase64Binary(Object value) {
        if (value instanceof String) {
            return ByteBuffer.wrap(Base64.getDecoder().decode((String)value));
        }
        if (value instanceof byte[]) {
            return ByteBuffer.wrap((byte[])value);
        }
        if (value instanceof ByteBuffer) {
            return (ByteBuffer)value;
        }
        throw new IllegalArgumentException("Cannot convert to binary: " + value.getClass().getName());
    }

    protected LocalDate convertDateValue(Object value) {
        if (value instanceof Number) {
            int days = ((Number)value).intValue();
            return DateTimeUtil.dateFromDays((int)days);
        }
        if (value instanceof String) {
            return LocalDate.parse((String)value);
        }
        if (value instanceof LocalDate) {
            return (LocalDate)value;
        }
        if (value instanceof Date) {
            int days = (int)(((Date)value).getTime() / 1000L / 60L / 60L / 24L);
            return DateTimeUtil.dateFromDays((int)days);
        }
        throw new ConnectException("Cannot convert date: " + String.valueOf(value));
    }

    protected LocalTime convertTimeValue(Object value) {
        if (value instanceof Number) {
            long millis = ((Number)value).longValue();
            return DateTimeUtil.timeFromMicros((long)(millis * 1000L));
        }
        if (value instanceof String) {
            return LocalTime.parse((String)value);
        }
        if (value instanceof LocalTime) {
            return (LocalTime)value;
        }
        if (value instanceof Date) {
            long millis = ((Date)value).getTime();
            return DateTimeUtil.timeFromMicros((long)(millis * 1000L));
        }
        throw new ConnectException("Cannot convert time: " + String.valueOf(value));
    }

    protected Temporal convertTimestampValue(Object value, Types.TimestampType type) {
        if (type.shouldAdjustToUTC()) {
            return this.convertOffsetDateTime(value);
        }
        return this.convertLocalDateTime(value);
    }

    private OffsetDateTime convertOffsetDateTime(Object value) {
        if (value instanceof Number) {
            long millis = ((Number)value).longValue();
            return DateTimeUtil.timestamptzFromMicros((long)(millis * 1000L));
        }
        if (value instanceof String) {
            return this.parseOffsetDateTime((String)value);
        }
        if (value instanceof OffsetDateTime) {
            return (OffsetDateTime)value;
        }
        if (value instanceof LocalDateTime) {
            return ((LocalDateTime)value).atOffset(ZoneOffset.UTC);
        }
        if (value instanceof Date) {
            return DateTimeUtil.timestamptzFromMicros((long)(((Date)value).getTime() * 1000L));
        }
        throw new ConnectException("Cannot convert timestamptz: " + String.valueOf(value) + ", type: " + String.valueOf(value.getClass()));
    }

    private OffsetDateTime parseOffsetDateTime(String str) {
        String tsStr = this.ensureTimestampFormat(str);
        try {
            return OFFSET_TIMESTAMP_FORMAT.parse((CharSequence)tsStr, OffsetDateTime::from);
        }
        catch (DateTimeParseException e) {
            return LocalDateTime.parse(tsStr, DateTimeFormatter.ISO_LOCAL_DATE_TIME).atOffset(ZoneOffset.UTC);
        }
    }

    private LocalDateTime convertLocalDateTime(Object value) {
        if (value instanceof Number) {
            long millis = ((Number)value).longValue();
            return DateTimeUtil.timestampFromMicros((long)(millis * 1000L));
        }
        if (value instanceof String) {
            return this.parseLocalDateTime((String)value);
        }
        if (value instanceof LocalDateTime) {
            return (LocalDateTime)value;
        }
        if (value instanceof OffsetDateTime) {
            return ((OffsetDateTime)value).toLocalDateTime();
        }
        if (value instanceof Date) {
            return DateTimeUtil.timestampFromMicros((long)(((Date)value).getTime() * 1000L));
        }
        throw new ConnectException("Cannot convert timestamp: " + String.valueOf(value) + ", type: " + String.valueOf(value.getClass()));
    }

    private LocalDateTime parseLocalDateTime(String str) {
        String tsStr = this.ensureTimestampFormat(str);
        try {
            return LocalDateTime.parse(tsStr, DateTimeFormatter.ISO_LOCAL_DATE_TIME);
        }
        catch (DateTimeParseException e) {
            return OFFSET_TIMESTAMP_FORMAT.parse((CharSequence)tsStr, OffsetDateTime::from).toLocalDateTime();
        }
    }

    private String ensureTimestampFormat(String str) {
        Object result = str;
        if (((String)result).charAt(10) == ' ') {
            result = ((String)result).substring(0, 10) + "T" + ((String)result).substring(11);
        }
        if (((String)result).length() > 22 && (((String)result).charAt(19) == '+' || ((String)result).charAt(19) == '-') && ((String)result).charAt(22) == ':') {
            result = ((String)result).substring(0, 19) + ((String)result).substring(19).replace(":", "");
        }
        return result;
    }
}

