/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.connect.channel;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.iceberg.AppendFiles;
import org.apache.iceberg.ContentFile;
import org.apache.iceberg.DataFile;
import org.apache.iceberg.DeleteFile;
import org.apache.iceberg.RowDelta;
import org.apache.iceberg.Snapshot;
import org.apache.iceberg.Table;
import org.apache.iceberg.catalog.Catalog;
import org.apache.iceberg.catalog.TableIdentifier;
import org.apache.iceberg.connect.IcebergSinkConfig;
import org.apache.iceberg.connect.channel.Channel;
import org.apache.iceberg.connect.channel.CommitState;
import org.apache.iceberg.connect.channel.Envelope;
import org.apache.iceberg.connect.channel.KafkaClientFactory;
import org.apache.iceberg.connect.events.CommitComplete;
import org.apache.iceberg.connect.events.CommitToTable;
import org.apache.iceberg.connect.events.DataWritten;
import org.apache.iceberg.connect.events.Event;
import org.apache.iceberg.connect.events.Payload;
import org.apache.iceberg.connect.events.StartCommit;
import org.apache.iceberg.connect.events.TableReference;
import org.apache.iceberg.exceptions.NoSuchTableException;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMap;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.util.Tasks;
import org.apache.iceberg.util.ThreadPools;
import org.apache.kafka.clients.admin.MemberDescription;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.sink.SinkTaskContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class Coordinator
extends Channel {
    private static final Logger LOG = LoggerFactory.getLogger(Coordinator.class);
    private static final ObjectMapper MAPPER = new ObjectMapper();
    private static final String COMMIT_ID_SNAPSHOT_PROP = "kafka.connect.commit-id";
    private static final String VALID_THROUGH_TS_SNAPSHOT_PROP = "kafka.connect.valid-through-ts";
    private static final Duration POLL_DURATION = Duration.ofSeconds(1L);
    private final Catalog catalog;
    private final IcebergSinkConfig config;
    private final int totalPartitionCount;
    private final String snapshotOffsetsProp;
    private final ExecutorService exec;
    private final CommitState commitState;
    private volatile boolean terminated;

    Coordinator(Catalog catalog, IcebergSinkConfig config, Collection<MemberDescription> members, KafkaClientFactory clientFactory, SinkTaskContext context) {
        super("coordinator", config.connectGroupId() + "-coord", config, clientFactory, context);
        this.catalog = catalog;
        this.config = config;
        this.totalPartitionCount = members.stream().mapToInt(desc -> desc.assignment().topicPartitions().size()).sum();
        this.snapshotOffsetsProp = String.format("kafka.connect.offsets.%s.%s", config.controlTopic(), config.connectGroupId());
        this.exec = ThreadPools.newFixedThreadPool((String)"iceberg-committer", (int)config.commitThreads());
        this.commitState = new CommitState(config);
    }

    void process() {
        if (this.commitState.isCommitIntervalReached()) {
            this.commitState.startNewCommit();
            Event event = new Event(this.config.connectGroupId(), (Payload)new StartCommit(this.commitState.currentCommitId()));
            this.send(event);
            LOG.info("Commit {} initiated", (Object)this.commitState.currentCommitId());
        }
        this.consumeAvailable(POLL_DURATION);
        if (this.commitState.isCommitTimedOut()) {
            this.commit(true);
        }
    }

    @Override
    protected boolean receive(Envelope envelope) {
        switch (envelope.event().payload().type()) {
            case DATA_WRITTEN: {
                this.commitState.addResponse(envelope);
                return true;
            }
            case DATA_COMPLETE: {
                this.commitState.addReady(envelope);
                if (this.commitState.isCommitReady(this.totalPartitionCount)) {
                    this.commit(false);
                }
                return true;
            }
        }
        return false;
    }

    private void commit(boolean partialCommit) {
        try {
            this.doCommit(partialCommit);
        }
        catch (Exception e) {
            LOG.warn("Commit failed, will try again next cycle", (Throwable)e);
        }
        finally {
            this.commitState.endCurrentCommit();
        }
    }

    private void doCommit(boolean partialCommit) {
        Map<TableReference, List<Envelope>> commitMap = this.commitState.tableCommitMap();
        String offsetsJson = this.offsetsJson();
        OffsetDateTime validThroughTs = this.commitState.validThroughTs(partialCommit);
        Tasks.foreach(commitMap.entrySet()).executeWith(this.exec).stopOnFailure().run(entry -> this.commitToTable((TableReference)entry.getKey(), (List)entry.getValue(), offsetsJson, validThroughTs));
        this.commitConsumerOffsets();
        this.commitState.clearResponses();
        Event event = new Event(this.config.connectGroupId(), (Payload)new CommitComplete(this.commitState.currentCommitId(), validThroughTs));
        this.send(event);
        LOG.info("Commit {} complete, committed to {} table(s), valid-through {}", new Object[]{this.commitState.currentCommitId(), commitMap.size(), validThroughTs});
    }

    private String offsetsJson() {
        try {
            return MAPPER.writeValueAsString(this.controlTopicOffsets());
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private void commitToTable(TableReference tableReference, List<Envelope> envelopeList, String offsetsJson, OffsetDateTime validThroughTs) {
        Table table;
        TableIdentifier tableIdentifier = tableReference.identifier();
        try {
            table = this.catalog.loadTable(tableIdentifier);
        }
        catch (NoSuchTableException e) {
            LOG.warn("Table not found, skipping commit: {}", (Object)tableIdentifier, (Object)e);
            return;
        }
        String branch = this.config.tableConfig(tableIdentifier.toString()).commitBranch();
        Map<Integer, Long> committedOffsets = this.lastCommittedOffsetsForTable(table, branch);
        List payloads = envelopeList.stream().filter(envelope -> {
            Long minOffset = (Long)committedOffsets.get(envelope.partition());
            return minOffset == null || envelope.offset() >= minOffset;
        }).map(envelope -> (DataWritten)envelope.event().payload()).collect(Collectors.toList());
        List<DataFile> dataFiles = payloads.stream().filter(payload -> payload.dataFiles() != null).flatMap(payload -> payload.dataFiles().stream()).filter(dataFile -> dataFile.recordCount() > 0L).filter(this.distinctByKey(ContentFile::location)).collect(Collectors.toList());
        List<DeleteFile> deleteFiles = payloads.stream().filter(payload -> payload.deleteFiles() != null).flatMap(payload -> payload.deleteFiles().stream()).filter(deleteFile -> deleteFile.recordCount() > 0L).filter(this.distinctByKey(ContentFile::location)).collect(Collectors.toList());
        if (this.terminated) {
            throw new ConnectException("Coordinator is terminated, commit aborted");
        }
        if (dataFiles.isEmpty() && deleteFiles.isEmpty()) {
            LOG.info("Nothing to commit to table {}, skipping", (Object)tableIdentifier);
        } else {
            if (deleteFiles.isEmpty()) {
                AppendFiles appendOp = table.newAppend();
                if (branch != null) {
                    appendOp.toBranch(branch);
                }
                appendOp.set(this.snapshotOffsetsProp, offsetsJson);
                appendOp.set(COMMIT_ID_SNAPSHOT_PROP, this.commitState.currentCommitId().toString());
                if (validThroughTs != null) {
                    appendOp.set(VALID_THROUGH_TS_SNAPSHOT_PROP, validThroughTs.toString());
                }
                dataFiles.forEach(arg_0 -> ((AppendFiles)appendOp).appendFile(arg_0));
                appendOp.commit();
            } else {
                RowDelta deltaOp = table.newRowDelta();
                if (branch != null) {
                    deltaOp.toBranch(branch);
                }
                deltaOp.set(this.snapshotOffsetsProp, offsetsJson);
                deltaOp.set(COMMIT_ID_SNAPSHOT_PROP, this.commitState.currentCommitId().toString());
                if (validThroughTs != null) {
                    deltaOp.set(VALID_THROUGH_TS_SNAPSHOT_PROP, validThroughTs.toString());
                }
                dataFiles.forEach(arg_0 -> ((RowDelta)deltaOp).addRows(arg_0));
                deleteFiles.forEach(arg_0 -> ((RowDelta)deltaOp).addDeletes(arg_0));
                deltaOp.commit();
            }
            Long snapshotId = this.latestSnapshot(table, branch).snapshotId();
            Event event = new Event(this.config.connectGroupId(), (Payload)new CommitToTable(this.commitState.currentCommitId(), tableReference, snapshotId, validThroughTs));
            this.send(event);
            LOG.info("Commit complete to table {}, snapshot {}, commit ID {}, valid-through {}", new Object[]{tableIdentifier, snapshotId, this.commitState.currentCommitId(), validThroughTs});
        }
    }

    private <T> Predicate<T> distinctByKey(Function<? super T, ?> keyExtractor) {
        ConcurrentMap seen = Maps.newConcurrentMap();
        return t -> seen.putIfAbsent(keyExtractor.apply(t), Boolean.TRUE) == null;
    }

    private Snapshot latestSnapshot(Table table, String branch) {
        if (branch == null) {
            return table.currentSnapshot();
        }
        return table.snapshot(branch);
    }

    private Map<Integer, Long> lastCommittedOffsetsForTable(Table table, String branch) {
        Snapshot snapshot = this.latestSnapshot(table, branch);
        while (snapshot != null) {
            Map summary = snapshot.summary();
            String value = (String)summary.get(this.snapshotOffsetsProp);
            if (value != null) {
                TypeReference<Map<Integer, Long>> typeRef = new TypeReference<Map<Integer, Long>>(){};
                try {
                    return (Map)MAPPER.readValue(value, (TypeReference)typeRef);
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            Long parentSnapshotId = snapshot.parentId();
            snapshot = parentSnapshotId != null ? table.snapshot(parentSnapshotId.longValue()) : null;
        }
        return ImmutableMap.of();
    }

    void terminate() {
        this.terminated = true;
        this.exec.shutdownNow();
        try {
            if (!this.exec.awaitTermination(1L, TimeUnit.MINUTES)) {
                throw new ConnectException("Timed out waiting for coordinator shutdown");
            }
        }
        catch (InterruptedException e) {
            throw new ConnectException("Interrupted while waiting for coordinator shutdown", (Throwable)e);
        }
    }
}

