/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.orc;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.iceberg.Schema;
import org.apache.iceberg.mapping.NameMapping;
import org.apache.iceberg.orc.ApplyNameMapping;
import org.apache.iceberg.orc.HasIds;
import org.apache.iceberg.orc.IdToOrcName;
import org.apache.iceberg.orc.OrcSchemaVisitor;
import org.apache.iceberg.orc.OrcToIcebergVisitor;
import org.apache.iceberg.orc.RemoveIds;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.relocated.com.google.common.collect.ImmutableMultimap;
import org.apache.iceberg.relocated.com.google.common.collect.Maps;
import org.apache.iceberg.types.Type;
import org.apache.iceberg.types.TypeUtil;
import org.apache.iceberg.types.Types;
import org.apache.orc.TypeDescription;

public final class ORCSchemaUtil {
    static final String ICEBERG_ID_ATTRIBUTE = "iceberg.id";
    static final String ICEBERG_REQUIRED_ATTRIBUTE = "iceberg.required";
    public static final String ICEBERG_BINARY_TYPE_ATTRIBUTE = "iceberg.binary-type";
    public static final String ICEBERG_LONG_TYPE_ATTRIBUTE = "iceberg.long-type";
    static final String ICEBERG_FIELD_LENGTH = "iceberg.length";
    static final String ICEBERG_STRUCT_TYPE_ATTRIBUTE = "iceberg.struct-type";
    static final String VARIANT = "VARIANT";
    static final String VARIANT_METADATA = "metadata";
    static final String VARIANT_VALUE = "value";
    static final String TIMESTAMP_UNIT = "iceberg.timestamp-unit";
    static final String MICROS = "MICROS";
    static final String NANOS = "NANOS";
    private static final ImmutableMultimap<Type.TypeID, TypeDescription.Category> TYPE_MAPPING = ImmutableMultimap.builder().put((Object)Type.TypeID.BOOLEAN, (Object)TypeDescription.Category.BOOLEAN).put((Object)Type.TypeID.INTEGER, (Object)TypeDescription.Category.BYTE).put((Object)Type.TypeID.INTEGER, (Object)TypeDescription.Category.SHORT).put((Object)Type.TypeID.INTEGER, (Object)TypeDescription.Category.INT).put((Object)Type.TypeID.LONG, (Object)TypeDescription.Category.LONG).put((Object)Type.TypeID.TIME, (Object)TypeDescription.Category.LONG).put((Object)Type.TypeID.FLOAT, (Object)TypeDescription.Category.FLOAT).put((Object)Type.TypeID.DOUBLE, (Object)TypeDescription.Category.DOUBLE).put((Object)Type.TypeID.DATE, (Object)TypeDescription.Category.DATE).put((Object)Type.TypeID.STRING, (Object)TypeDescription.Category.CHAR).put((Object)Type.TypeID.STRING, (Object)TypeDescription.Category.VARCHAR).put((Object)Type.TypeID.STRING, (Object)TypeDescription.Category.STRING).put((Object)Type.TypeID.UUID, (Object)TypeDescription.Category.BINARY).put((Object)Type.TypeID.FIXED, (Object)TypeDescription.Category.BINARY).put((Object)Type.TypeID.BINARY, (Object)TypeDescription.Category.BINARY).put((Object)Type.TypeID.DECIMAL, (Object)TypeDescription.Category.DECIMAL).build();

    private ORCSchemaUtil() {
    }

    public static TypeDescription convert(Schema schema) {
        TypeDescription root = TypeDescription.createStruct();
        Types.StructType schemaRoot = schema.asStruct();
        for (Types.NestedField field : schemaRoot.asStructType().fields()) {
            TypeDescription orcColumnType = ORCSchemaUtil.convert(field.fieldId(), field.type(), field.isRequired());
            if (orcColumnType == null) continue;
            root.addField(field.name(), orcColumnType);
        }
        return root;
    }

    private static TypeDescription convert(Integer fieldId, Type type, boolean isRequired) {
        TypeDescription orcType;
        switch (type.typeId()) {
            case UNKNOWN: {
                return null;
            }
            case BOOLEAN: {
                orcType = TypeDescription.createBoolean();
                break;
            }
            case INTEGER: {
                orcType = TypeDescription.createInt();
                break;
            }
            case TIME: {
                orcType = TypeDescription.createLong();
                orcType.setAttribute(ICEBERG_LONG_TYPE_ATTRIBUTE, LongType.TIME.toString());
                break;
            }
            case LONG: {
                orcType = TypeDescription.createLong();
                orcType.setAttribute(ICEBERG_LONG_TYPE_ATTRIBUTE, LongType.LONG.toString());
                break;
            }
            case FLOAT: {
                orcType = TypeDescription.createFloat();
                break;
            }
            case DOUBLE: {
                orcType = TypeDescription.createDouble();
                break;
            }
            case DATE: {
                orcType = TypeDescription.createDate();
                break;
            }
            case TIMESTAMP: {
                Types.TimestampType tsType = (Types.TimestampType)type;
                orcType = tsType.shouldAdjustToUTC() ? TypeDescription.createTimestampInstant() : TypeDescription.createTimestamp();
                orcType.setAttribute(TIMESTAMP_UNIT, MICROS);
                break;
            }
            case TIMESTAMP_NANO: {
                Types.TimestampNanoType tsNanoType = (Types.TimestampNanoType)type;
                orcType = tsNanoType.shouldAdjustToUTC() ? TypeDescription.createTimestampInstant() : TypeDescription.createTimestamp();
                orcType.setAttribute(TIMESTAMP_UNIT, NANOS);
                break;
            }
            case STRING: {
                orcType = TypeDescription.createString();
                break;
            }
            case UUID: {
                orcType = TypeDescription.createBinary();
                orcType.setAttribute(ICEBERG_BINARY_TYPE_ATTRIBUTE, BinaryType.UUID.toString());
                break;
            }
            case FIXED: {
                orcType = TypeDescription.createBinary();
                orcType.setAttribute(ICEBERG_BINARY_TYPE_ATTRIBUTE, BinaryType.FIXED.toString());
                orcType.setAttribute(ICEBERG_FIELD_LENGTH, Integer.toString(((Types.FixedType)type).length()));
                break;
            }
            case BINARY: {
                orcType = TypeDescription.createBinary();
                orcType.setAttribute(ICEBERG_BINARY_TYPE_ATTRIBUTE, BinaryType.BINARY.toString());
                break;
            }
            case DECIMAL: {
                Types.DecimalType decimal = (Types.DecimalType)type;
                orcType = TypeDescription.createDecimal().withScale(decimal.scale()).withPrecision(decimal.precision());
                break;
            }
            case VARIANT: {
                orcType = TypeDescription.createStruct();
                orcType.addField(VARIANT_METADATA, TypeDescription.createBinary());
                orcType.addField(VARIANT_VALUE, TypeDescription.createBinary());
                orcType.setAttribute(ICEBERG_STRUCT_TYPE_ATTRIBUTE, VARIANT);
                break;
            }
            case STRUCT: {
                orcType = TypeDescription.createStruct();
                for (Types.NestedField field : type.asStructType().fields()) {
                    TypeDescription childType = ORCSchemaUtil.convert(field.fieldId(), field.type(), field.isRequired());
                    orcType.addField(field.name(), childType);
                }
                break;
            }
            case LIST: {
                Types.ListType list = (Types.ListType)type;
                TypeDescription elementType = ORCSchemaUtil.convert(list.elementId(), list.elementType(), list.isElementRequired());
                orcType = TypeDescription.createList((TypeDescription)elementType);
                break;
            }
            case MAP: {
                Types.MapType map = (Types.MapType)type;
                TypeDescription keyType = ORCSchemaUtil.convert(map.keyId(), map.keyType(), true);
                TypeDescription valueType = ORCSchemaUtil.convert(map.valueId(), map.valueType(), map.isValueRequired());
                orcType = TypeDescription.createMap((TypeDescription)keyType, (TypeDescription)valueType);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unhandled type " + String.valueOf(type.typeId()));
            }
        }
        orcType.setAttribute(ICEBERG_ID_ATTRIBUTE, String.valueOf(fieldId));
        orcType.setAttribute(ICEBERG_REQUIRED_ATTRIBUTE, String.valueOf(isRequired));
        return orcType;
    }

    public static Schema convert(TypeDescription orcSchema) {
        List children = orcSchema.getChildren();
        List childrenNames = orcSchema.getFieldNames();
        Preconditions.checkState((children.size() == childrenNames.size() ? 1 : 0) != 0, (Object)"Error in ORC file, children fields and names do not match.");
        OrcToIcebergVisitor schemaConverter = new OrcToIcebergVisitor();
        List fields = OrcToIcebergVisitor.visitSchema(orcSchema, schemaConverter).stream().filter(Optional::isPresent).map(Optional::get).collect(Collectors.toList());
        if (fields.isEmpty()) {
            throw new IllegalArgumentException("ORC schema does not contain Iceberg IDs");
        }
        return new Schema(fields);
    }

    public static TypeDescription buildOrcProjection(Schema schema, TypeDescription originalOrcSchema) {
        Map<Integer, OrcField> icebergToOrc = ORCSchemaUtil.icebergToOrcMapping("root", originalOrcSchema);
        return ORCSchemaUtil.buildOrcProjection(schema, Integer.MIN_VALUE, (Type)schema.asStruct(), true, icebergToOrc);
    }

    private static TypeDescription buildOrcProjection(Schema root, Integer fieldId, Type type, boolean isRequired, Map<Integer, OrcField> mapping) {
        TypeDescription orcType;
        switch (type.typeId()) {
            case STRUCT: {
                orcType = TypeDescription.createStruct();
                for (Types.NestedField nestedField : type.asStructType().fields()) {
                    String name = Optional.ofNullable(mapping.get(nestedField.fieldId())).map(OrcField::name).orElseGet(() -> nestedField.name() + "_r" + nestedField.fieldId());
                    TypeDescription childType = ORCSchemaUtil.buildOrcProjection(root, nestedField.fieldId(), nestedField.type(), isRequired && nestedField.isRequired(), mapping);
                    if (childType == null) continue;
                    orcType.addField(name, childType);
                }
                break;
            }
            case LIST: {
                Types.ListType list = (Types.ListType)type;
                TypeDescription elementType = ORCSchemaUtil.buildOrcProjection(root, list.elementId(), list.elementType(), isRequired && list.isElementRequired(), mapping);
                Preconditions.checkArgument((elementType != null ? 1 : 0) != 0, (Object)"Invalid element type: unknown");
                orcType = TypeDescription.createList((TypeDescription)elementType);
                break;
            }
            case MAP: {
                Types.MapType map = (Types.MapType)type;
                TypeDescription keyType = ORCSchemaUtil.buildOrcProjection(root, map.keyId(), map.keyType(), isRequired, mapping);
                TypeDescription valueType = ORCSchemaUtil.buildOrcProjection(root, map.valueId(), map.valueType(), isRequired && map.isValueRequired(), mapping);
                Preconditions.checkArgument((keyType != null ? 1 : 0) != 0, (Object)"Invalid key type: unknown");
                Preconditions.checkArgument((valueType != null ? 1 : 0) != 0, (Object)"Invalid value type: unknown");
                orcType = TypeDescription.createMap((TypeDescription)keyType, (TypeDescription)valueType);
                break;
            }
            case VARIANT: {
                orcType = TypeDescription.createStruct();
                orcType.addField(VARIANT_METADATA, TypeDescription.createBinary());
                orcType.addField(VARIANT_VALUE, TypeDescription.createBinary());
                orcType.setAttribute(ICEBERG_STRUCT_TYPE_ATTRIBUTE, VARIANT);
                break;
            }
            default: {
                if (mapping.containsKey(fieldId)) {
                    TypeDescription originalType = mapping.get(fieldId).type();
                    Optional<TypeDescription> promotedType = ORCSchemaUtil.getPromotedType(type, originalType);
                    if (promotedType.isPresent()) {
                        orcType = promotedType.get();
                        break;
                    }
                    Preconditions.checkArgument((boolean)ORCSchemaUtil.isSameType(originalType, type), (String)"Can not promote %s type to %s", (Object)originalType.getCategory(), (Object)type.typeId().name());
                    orcType = originalType.clone();
                    break;
                }
                Types.NestedField field = root.findField(fieldId.intValue());
                if (isRequired) {
                    Preconditions.checkArgument((field.initialDefault() != null ? 1 : 0) != 0, (String)"Missing required field: %s (%s)", (Object)root.findColumnName(fieldId.intValue()), (Object)type);
                }
                if (field.initialDefault() != null) {
                    throw new UnsupportedOperationException(String.format("ORC cannot read default value for field %s (%s): %s", root.findColumnName(fieldId.intValue()), type, field.initialDefault()));
                }
                orcType = ORCSchemaUtil.convert(fieldId, type, false);
            }
        }
        if (orcType != null) {
            orcType.setAttribute(ICEBERG_ID_ATTRIBUTE, fieldId.toString());
        }
        return orcType;
    }

    private static Map<Integer, OrcField> icebergToOrcMapping(String name, TypeDescription orcType) {
        HashMap icebergToOrc = Maps.newHashMap();
        switch (orcType.getCategory()) {
            case STRUCT: {
                List childrenNames = orcType.getFieldNames();
                List children = orcType.getChildren();
                for (int i = 0; i < children.size(); ++i) {
                    icebergToOrc.putAll(ORCSchemaUtil.icebergToOrcMapping((String)childrenNames.get(i), (TypeDescription)children.get(i)));
                }
                break;
            }
            case LIST: {
                icebergToOrc.putAll(ORCSchemaUtil.icebergToOrcMapping("element", (TypeDescription)orcType.getChildren().get(0)));
                break;
            }
            case MAP: {
                icebergToOrc.putAll(ORCSchemaUtil.icebergToOrcMapping("key", (TypeDescription)orcType.getChildren().get(0)));
                icebergToOrc.putAll(ORCSchemaUtil.icebergToOrcMapping(VARIANT_VALUE, (TypeDescription)orcType.getChildren().get(1)));
            }
        }
        if (orcType.getId() > 0) {
            ORCSchemaUtil.icebergID(orcType).ifPresent(integer -> icebergToOrc.put(integer, new OrcField(name, orcType)));
        }
        return icebergToOrc;
    }

    private static Optional<TypeDescription> getPromotedType(Type icebergType, TypeDescription originalOrcType) {
        Types.DecimalType newDecimal;
        TypeDescription promotedOrcType = null;
        if (Type.TypeID.LONG.equals((Object)icebergType.typeId()) && TypeDescription.Category.INT.equals((Object)originalOrcType.getCategory())) {
            promotedOrcType = TypeDescription.createLong();
        } else if (Type.TypeID.DOUBLE.equals((Object)icebergType.typeId()) && TypeDescription.Category.FLOAT.equals((Object)originalOrcType.getCategory())) {
            promotedOrcType = TypeDescription.createDouble();
        } else if (Type.TypeID.DECIMAL.equals((Object)icebergType.typeId()) && TypeDescription.Category.DECIMAL.equals((Object)originalOrcType.getCategory()) && (newDecimal = (Types.DecimalType)icebergType).scale() == originalOrcType.getScale() && newDecimal.precision() > originalOrcType.getPrecision()) {
            promotedOrcType = TypeDescription.createDecimal().withScale(newDecimal.scale()).withPrecision(newDecimal.precision());
        }
        return Optional.ofNullable(promotedOrcType);
    }

    private static boolean isSameType(TypeDescription orcType, Type icebergType) {
        if (icebergType.typeId() == Type.TypeID.TIMESTAMP) {
            Types.TimestampType tsType = (Types.TimestampType)icebergType;
            return Objects.equals(tsType.shouldAdjustToUTC() ? TypeDescription.Category.TIMESTAMP_INSTANT : TypeDescription.Category.TIMESTAMP, orcType.getCategory());
        }
        if (icebergType.typeId() == Type.TypeID.TIMESTAMP_NANO) {
            Types.TimestampNanoType tsType = (Types.TimestampNanoType)icebergType;
            return Objects.equals(tsType.shouldAdjustToUTC() ? TypeDescription.Category.TIMESTAMP_INSTANT : TypeDescription.Category.TIMESTAMP, orcType.getCategory());
        }
        return TYPE_MAPPING.containsEntry((Object)icebergType.typeId(), (Object)orcType.getCategory());
    }

    static Optional<Integer> icebergID(TypeDescription orcType) {
        return Optional.ofNullable(orcType.getAttributeValue(ICEBERG_ID_ATTRIBUTE)).map(Integer::parseInt);
    }

    public static int fieldId(TypeDescription orcType) {
        String idStr = orcType.getAttributeValue(ICEBERG_ID_ATTRIBUTE);
        Preconditions.checkNotNull((Object)idStr, (String)"Missing expected '%s' property", (Object)ICEBERG_ID_ATTRIBUTE);
        return Integer.parseInt(idStr);
    }

    static boolean isOptional(TypeDescription orcType) {
        String isRequiredStr = orcType.getAttributeValue(ICEBERG_REQUIRED_ATTRIBUTE);
        if (isRequiredStr != null) {
            return !Boolean.parseBoolean(isRequiredStr);
        }
        return true;
    }

    static TypeDescription removeIds(TypeDescription type) {
        return OrcSchemaVisitor.visit(type, new RemoveIds());
    }

    static boolean hasIds(TypeDescription orcSchema) {
        return OrcSchemaVisitor.visit(orcSchema, new HasIds());
    }

    static TypeDescription applyNameMapping(TypeDescription orcSchema, NameMapping nameMapping) {
        return OrcSchemaVisitor.visit(orcSchema, new ApplyNameMapping(nameMapping));
    }

    public static Map<Integer, String> idToOrcName(Schema schema) {
        return (Map)TypeUtil.visit((Schema)schema, (TypeUtil.SchemaVisitor)new IdToOrcName());
    }

    public static enum LongType {
        TIME,
        LONG;

    }

    public static enum BinaryType {
        UUID,
        FIXED,
        BINARY;

    }

    private static class OrcField {
        private final String name;
        private final TypeDescription type;

        OrcField(String name, TypeDescription type) {
            this.name = name;
            this.type = type;
        }

        public String name() {
            return this.name;
        }

        public TypeDescription type() {
            return this.type;
        }
    }
}

